%% Script to analyze foci distances from microscopy images

% Assumes the following channel order:
%   c1 - phase contrast
%   c2 - fluo1 - ori 
%   c3 - fluo2 - ter 
%   c4 - fluo3 - MukB


%% 1. Run superSegger analysis
superSeggerGui

%% 2. Run estimateAlignment to align fluorescence channels to phase contrast
%   If there is slight shift between channels. This creates a alingment
%   vector. Assumes random orientation of cells.

% set Matlab path to the superSegger folder
estimateAlignment

%% 3. Extract information from superSegger folder structure
%   This generates a single *_pop_spots.mat file with all cells from
%   different XY.

% set Matlab path to the superSegger folder
load('imAlign.mat')

% Use this for spot detection of MukB
generateSpotsforDist(0, out)

% Use this for brightest pixel detection of MukB
generateSpotsforDist(2, out)

%% 4A. MukB distance to closest ori/ter using spot detection for MukB for 1 ori/ 1 ter cells
% chooses single ori and ter cells to avoid copy number bias
% optimal distance would be diffraction limit of ypet (530nm / 2 = 265nm)
% pixel size is 110nm -> 265nm/110nm = 2.4 px

pixel = 0.11;               % pixel size in um
spot_threshold = 3;         % minimum score for a spot
spot_sigma = [0.7 1.1];     % MukBEF spot width
distance_threshold = 2.4;   % 265 nm


% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

ori_dist = [];          % distance to ori vector
ter_dist = [];          % distance to ter vector
ori_dist_all = [];      % combined distances to ori vector
ter_dist_all = [];      % combined distances to ter vector
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    for ii = 1:length(spots)
        % cells that have right number of ori/ter and MukB spot(s)
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                length(spots(ii).fluo1.l) == 1 && ...
                ~isempty(spots(ii).fluo2) && ~isempty(spots(ii).fluo2.l) && ...
                length(spots(ii).fluo2.l) == 1 && ...
                ~isempty(spots(ii).fluo3) && ~isempty(spots(ii).fluo3.l)
                
            % mask for good enough spots
            mask = spots(ii).fluo3.score > spot_threshold & ...
                spots(ii).fluo3.sigma > spot_sigma(1) & ...
                spots(ii).fluo3.sigma < spot_sigma(2);
            
            if sum(mask) > 0
            
                % MukB coord
                xcoord = spots(ii).fluo3.coord(1:2:end);
                ycoord = spots(ii).fluo3.coord(2:2:end);
                X2 = [xcoord(mask)' ycoord(mask)'];

                % ori
                xcoord = spots(ii).fluo1.coord(1:2:end);
                ycoord = spots(ii).fluo1.coord(2:2:end);
                X1 = [xcoord' ycoord'];
                X = [X1; X2];
                d = pdist(X,'euclidean');
                Z = squareform(d);
                Z(Z == 0) = NaN;
                ori_dist = [ori_dist min(Z(1,:))];

                % ter
                xcoord = spots(ii).fluo2.coord(1:2:end);
                ycoord = spots(ii).fluo2.coord(2:2:end);
                X1 = [xcoord' ycoord'];
                X = [X1; X2];
                d = pdist(X,'euclidean');
                Z = squareform(d);
                Z(Z == 0) = NaN;
                ter_dist = [ter_dist min(Z(1,:))];
            end
        end
    end

% mean distances
fprintf('\n')
disp(['# Results ' num2str(kk) ' #'])
disp(['MukB-ori dists - mean = ' num2str((mean(ori_dist.*pixel))) ' um, no. cells = ' num2str(length(ori_dist))])
disp(['MukB-ter dists - mean = ' num2str((mean(ter_dist.*pixel))) ' um, no. cells = ' num2str(length(ter_dist))])

% percentage of spots within a distance from ori/ter
ori_precentage = sum(ori_dist < distance_threshold)./length(ori_dist);
disp(['Percentage of MukB spots within ' num2str(distance_threshold) ' px ('...
    num2str(distance_threshold.*pixel.*1000) ' nm) from ori is ' ...
    num2str(ori_precentage.*100) '% (' num2str(length(ori_dist)) ' cells)' ])
ter_precentage = sum(ter_dist < distance_threshold)./length(ter_dist);
disp(['Percentage of MukB spots within ' num2str(distance_threshold) ' px ('...
    num2str(distance_threshold.*pixel.*1000) ' nm) from ter is ' ...
    num2str(ter_precentage.*100) '% (' num2str(length(ter_dist)) ' cells)' ])
fprintf('\n')

% collect results from different experiments
ori_dist_all = [ori_dist_all ori_dist];
ter_dist_all = [ter_dist_all ter_dist]; 
end

% CDFs
figure; hold on;
[f,x,flo,fup] = ecdf(ori_dist_all.*pixel);
plot(x,f,'r','LineWidth',1.5)
[f,x,flo,fup] = ecdf(ter_dist_all.*pixel);
plot(x,f,'b','LineWidth',1.5)
legend('MukB-ori','MukB-ter')
xlabel('Distance (\mum)')
ylabel('CDF')

% PDFs
binning = 0.05;  % histogram binning every x um

[h1, I1] = hist(ori_dist_all.*pixel,0:binning:max(ori_dist_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ori')
xlabel('Distance (\mum)')
[h1, I1] = hist(ter_dist_all.*pixel,0:binning:max(ter_dist_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ter')
xlabel('Distance (\mum)')

%% 4B. Distance from each MukBEF spot to the closest ori1/ter3
% optimal distance would be diffraction limit of ypet (530nm / 2 = 265nm)
% pixel size is 110nm -> 265nm/110nm = 2.4 px

pixel = 0.11;                   % pixel size in um
no_ori = [1 2 3 4];             % only cells with X number of ori as a vector
no_ter = [1 2 3 4];             % only cells with X number of ter as a vector
no_MukB = [1 2 3 4];            % only cells with X number of MukBEF as a vector
spot_threshold = 3;             % minimum score for MukBEF spot
spot_sigma = [0.7 1.1];         % MukBEF spot width
distance_threshold = 2.4;       % 265 nm, similar to 2 px in Nolivos et al.

% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

ori_dist = [];          % distance to ori vector
ter_dist = [];          % distance to ter vector
ori_dist_all = [];      % combined distances to ori vector
ter_dist_all = [];      % combined distances to ter vector
no_cells_all = [];      % combined number of cells
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    no_cells = 0;
    for ii = 1:length(spots)
        % cells that have right number of ori/ter and MukB spot(s)
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                sum(length(spots(ii).fluo1.l) == no_ori) > 0 && ...
                ~isempty(spots(ii).fluo2) && ~isempty(spots(ii).fluo2.l) && ...
                sum(length(spots(ii).fluo2.l) == no_ter) > 0 && ...
                ~isempty(spots(ii).fluo3) && ~isempty(spots(ii).fluo3.l)
                
            % mask for good enough spots
            mask = spots(ii).fluo3.score > spot_threshold & ...
                spots(ii).fluo3.sigma > spot_sigma(1) & ...
                spots(ii).fluo3.sigma < spot_sigma(2);
            
            % only cells with right number of MukBEF
            if sum(sum(mask) == no_MukB) > 0
            
                % MukB coord
                xcoord = spots(ii).fluo3.coord(1:2:end);
                ycoord = spots(ii).fluo3.coord(2:2:end);
                X2 = [xcoord(mask)' ycoord(mask)'];

                % ori
                xcoord = spots(ii).fluo1.coord(1:2:end);
                ycoord = spots(ii).fluo1.coord(2:2:end);
                X1 = [xcoord' ycoord'];
                X = [X2; X1];
                d = pdist(X,'euclidean');
                Z = squareform(d);
                Z(Z == 0) = NaN;
                
                % loop over MukBEF spot minimum distances
                for jj = 1:sum(mask)
                    ori_dist = [ori_dist min(Z(jj,:))];
                end

                % ter
                xcoord = spots(ii).fluo2.coord(1:2:end);
                ycoord = spots(ii).fluo2.coord(2:2:end);
                X1 = [xcoord' ycoord'];
                X = [X2; X1];
                d = pdist(X,'euclidean');
                Z = squareform(d);
                Z(Z == 0) = NaN;
                
                % loop over MukBEF spot minimum distances
                for jj = 1:sum(mask)
                    ter_dist = [ter_dist min(Z(jj,:))];
                end
                
                no_cells = no_cells + 1;
            end
        end
    end

% mean distances
fprintf('\n')
disp(['# Results ' num2str(kk) ' #'])
disp(['MukB-ori dists - mean = ' num2str((mean(ori_dist.*pixel))) ' um, no. cells = ' ...
    num2str(no_cells) ', no. distances = ' num2str(length(ori_dist))])
disp(['MukB-ter dists - mean = ' num2str((mean(ter_dist.*pixel))) ' um, no. cells = ' ...
    num2str(no_cells) ', no. distances = ' num2str(length(ter_dist))])

% percentage of spots within a distance from ori/ter
ori_precentage = sum(ori_dist < distance_threshold)./length(ori_dist);
disp(['Percentage of MukB spots within ' num2str(distance_threshold) ' px ('...
    num2str(distance_threshold.*pixel.*1000) ' nm) from ori is ' ...
    num2str(ori_precentage.*100) '% (' num2str(no_cells) ' cells)' ])
ter_precentage = sum(ter_dist < distance_threshold)./length(ter_dist);
disp(['Percentage of MukB spots within ' num2str(distance_threshold) ' px ('...
    num2str(distance_threshold.*pixel.*1000) ' nm) from ter is ' ...
    num2str(ter_precentage.*100) '% (' num2str(no_cells) ' cells)' ])
fprintf('\n')

% collect results from different experiments
ori_dist_all = [ori_dist_all ori_dist];
ter_dist_all = [ter_dist_all ter_dist]; 
no_cells_all = [no_cells_all no_cells];
end

no_cells_sum = sum(no_cells_all);

% CDFs
figure; hold on;
[f,x,flo,fup] = ecdf(ori_dist_all.*pixel);
plot(x,f,'r','LineWidth',1.5)
[f,x,flo,fup] = ecdf(ter_dist_all.*pixel);
plot(x,f,'b','LineWidth',1.5)
legend('MukB-ori','MukB-ter')
xlabel('Distance (\mum)')
ylabel('CDF')

% PDFs
binning = 0.05;  % histogram binning every x um

[h1, I1] = hist(ori_dist_all.*pixel,0:binning:max(ori_dist_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ori')
xlabel('Distance (\mum)')
[h1, I1] = hist(ter_dist_all.*pixel,0:binning:max(ter_dist_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ter')
xlabel('Distance (\mum)')

%% 4C. Brightest MukB pixel distance to closest ori/ter
% The script chooses single ori and ter cells to avoid copy number bias

pixel = 0.11;               % pixel size in um
no_ori = [1];               % only cells with X number of ori e.g. [1 2]
no_ter = [1];               % only cells with X number of ter e.g. [1 2]

% select *_pop_bright.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

ori_dist_bright = [];           % distance to brightest px
ter_dist_bright = [];           % distance to brightest px
ori_dist_bright_all = [];       % combined distances to ori vector
ter_dist_bright_all = [];       % combined distances to ter vector
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    for ii = 1:length(spots)
        % cells that have same number of ori/ter
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                sum(length(spots(ii).fluo1.l) == no_ori) > 0 && ...
                ~isempty(spots(ii).fluo2) && ~isempty(spots(ii).fluo2.l) && ...
                sum(length(spots(ii).fluo2.l) == no_ter) > 0 && ...
                ~isempty(spots(ii).fluo3A.coord)
                
            % brightest MukB px
            Ints = double(spots(ii).fluo3A.Int);
            [~,I] = max(Ints);
            X2 = spots(ii).fluo3A.coord(I,:);

            % ori
            xcoord = spots(ii).fluo1.coord(1:2:end);
            ycoord = spots(ii).fluo1.coord(2:2:end);
            X1 = [xcoord' ycoord'];
            X = [X2; X1];
            d = pdist(X,'euclidean');
            Z = squareform(d);
            Z(Z == 0) = NaN;
            ori_dist_bright = [ori_dist_bright min(min(Z))];

            % ter
            xcoord = spots(ii).fluo2.coord(1:2:end);
            ycoord = spots(ii).fluo2.coord(2:2:end);
            X1 = [xcoord' ycoord'];
            X = [X2; X1];
            d = pdist(X,'euclidean');
            Z = squareform(d);
            Z(Z == 0) = NaN;
            ter_dist_bright = [ter_dist_bright min(min(Z))];
        end
    end

% mean distances
fprintf('\n')
disp(['MukB-ori dists - mean = ' num2str((mean(ori_dist_bright.*pixel))) ' um, no. cells = ' num2str(length(ori_dist_bright))])
disp(['MukB-ter dists - mean = ' num2str((mean(ter_dist_bright.*pixel))) ' um, no. cells = ' num2str(length(ter_dist_bright))])
fprintf('\n')

% collect results from different experiments
ori_dist_bright_all = [ori_dist_bright_all ori_dist_bright];      
ter_dist_bright_all = [ter_dist_bright_all ter_dist_bright]; 
end

% CDFs
figure; hold on;
[f,x,flo,fup] = ecdf(ori_dist_bright_all.*pixel);
plot(x,f,'r','LineWidth',1.5)
[f,x,flo,fup] = ecdf(ter_dist_bright_all.*pixel);
plot(x,f,'b','LineWidth',1.5)
legend('MukB-ori','MukB-ter')
xlabel('Distance (\mum)')

% PDFs
binning = 0.05;  % histogram binning every x um

[h1, I1] = hist(ori_dist_bright_all.*pixel,0:binning:max(ori_dist_bright_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ori')
xlabel('Distance (\mum)')
[h1, I1] = hist(ter_dist_bright_all.*pixel,0:binning:max(ter_dist_bright_all.*pixel));
h1 = h1./sum(h1);
figure; bar(I1,h1)
title('MukB-ter')
xlabel('Distance (\mum)')

%% Cell length and width from all cells

pixel = 0.11;           % pixel size in um

% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

lengths_all = [];       % combined cell lengths
widths_all = [];        % combined cell widths
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    lengths_all = [lengths_all [spots.length].*pixel];
    widths_all = [widths_all [spots.width].*pixel];
end

% box plots
figure; boxplot(lengths_all)
title('cell lengths')
ylabel('Cell length (\mum)')
figure; boxplot(widths_all)
title('cell widths')
ylabel('Cell width (\mum)')

%% Number of MukB spots
% prints spot statistics
% can select multiple files from same folder

% parameters
threshold = 3;    % score threshold for spots (minimum 2.5)

% select *_pop_bright.mat file(s)

%(A) all possible combinations of ori and ter numbers
no_spots = calc_spot_numbers(threshold);

%(B) specific number of ori and ter in [ ] with spaces 
% e.g. [1] is only two, [1 2] is either one or two
no_ori = [2 3 4];   % number of ori in cells
no_ter = [2 3 4];   % number of ter in cells
no_spots = calc_spot_numbers(threshold, no_ori, no_ter);


% plot histogram of spots numbers
figure; hold on
[N, X] = hist(no_spots, 0:max(no_spots));
N = N./sum(N);
bar(X,N)
xlabel('Number of MukBEF foci')
ylabel('Fraction of cells')

%% ori1 localization along normalized long cell axis

no_ori = [2];               % cells with X number of ori, e.g. [1 2]
no_ter = [1 2 3];           % cells with X number of ori, e.g. [1 2]
binning = 0.01;             % bar width in histogram

% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

norm_dist_all = [];         % combined cell lengths
dist_all_means = [];        
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    norm_dist = [];
    for ii = 1:length(spots)
        % specific number of ori and ter
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                sum(length(spots(ii).fluo1.l) == no_ori) > 0 && ...
                ~isempty(spots(ii).fluo2) && ~isempty(spots(ii).fluo2.l) && ...
                sum(length(spots(ii).fluo2.l) == no_ter) > 0
            
            norm_dist = [norm_dist spots(ii).fluo1.l./spots(ii).length];
        end
    end
    norm_dist_all = [norm_dist_all norm_dist];
    dist_all_means = [dist_all_means mean(abs(norm_dist))];
    
    % mean distances on negative and positive side
    fprintf('\n')
    disp(['# results ' num2str(kk) ' #'])
    disp(['Mean normalized distance from cell center - ' num2str((mean(abs(norm_dist)))) ...
        ', no. cells = ' num2str(length(norm_dist))])
    fprintf('\n')
end

fprintf('\n')
disp('# All results #')
disp(['Mean of repeats - ' num2str(mean(dist_all_means)) ...
    ', std ' num2str(std(dist_all_means)) ...
    ', no. cells = ' num2str(length(norm_dist_all))])
fprintf('\n')

% plot histogram of normalized ori1 localization (-0.5 to 0.5)
figure;
[N, X] = hist(norm_dist_all, -0.5:binning:0.5);
N = N./sum(N);
bar(X,N)
xlabel('Normalized long cell axis')
ylabel('Probability density')

figure; plot(X,N,'LineWidth',1)
xlabel('Normalized long cell axis')
ylabel('Probability density')

%% ter3 localization along normalized long cell axis

no_ori = [1];       	% cells with X number of ori, e.g. [1 2]
no_ter = [1];           % cells with X number of ori, e.g. [1 2]
binning = 0.01;         % bar width in histogram

% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

norm_dist_all = [];         % combined cell lengths
dist_all_means = [];        
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    norm_dist = [];
    for ii = 1:length(spots)
        % specific number of ori and ter
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                sum(length(spots(ii).fluo1.l) == no_ori) > 0 && ...
                ~isempty(spots(ii).fluo2) && ~isempty(spots(ii).fluo2.l) && ...
                sum(length(spots(ii).fluo2.l) == no_ter) > 0
            
            norm_dist = [norm_dist spots(ii).fluo2.l./spots(ii).length];
        end
    end
    norm_dist_all = [norm_dist_all norm_dist];
    dist_all_means = [dist_all_means mean(abs(norm_dist))];
    
    % mean distances on negative and positive side
    fprintf('\n')
    disp(['# results ' num2str(kk) ' #'])
    disp(['Mean normalized distance from cell center - ' num2str((mean(abs(norm_dist)))) ...
        ', no. cells = ' num2str(length(norm_dist))])
    fprintf('\n')
end

fprintf('\n')
disp('# All results #')
disp(['Mean of repeats - ' num2str(mean(dist_all_means)) ...
    ', std ' num2str(std(dist_all_means)) ...
    ', no. cells = ' num2str(length(norm_dist_all))])
fprintf('\n')

% plot histogram of normalized ori1 localization (-0.5 to 0.5)
figure;
[N, X] = hist(norm_dist_all, -0.5:binning:0.5);
N = N./sum(N);
bar(X,N)
xlabel('Normalized long cell axis')
ylabel('Probability density')

figure; plot(X,N,'LineWidth',1)
xlabel('Normalized long cell axis')
ylabel('Probability density')


%% MukBEF localization along normalized long cell axis

no_ori = [2];               % cells with X number of ori, e.g. [1 2] or [2]
no_MukB = [1 2 3 4];      % cells with X number of MukB foci e.g. [1 2 3 4 5] 
binning = 0.01;             % bar width in histogram
thresSigma = [0.7 1.1];     % MukBEF spot width
thresScore = 3;             % MukBEF spot score threshold


% select *_pop_spots.mat file(s)
[filename,pathname] = uigetfile('*.mat', 'Select','MultiSelect', 'on');
if iscell(filename)
    no_files = length(filename);
    filename_cell = filename;
else
    no_files = 1;
    filename_cell{1} = filename;
end

norm_dist_all = [];         % combined cell lengths
dist_all_means = [];        
for kk = 1:no_files
    load([pathname filename_cell{kk}]);
    norm_dist = [];
    for ii = 1:length(spots)
        if ~isempty(spots(ii).fluo1) && ~isempty(spots(ii).fluo1.l) && ...
                sum(length(spots(ii).fluo1.l) == no_ori) > 0 && ...
                ~isempty(spots(ii).fluo3) && ~isempty(spots(ii).fluo3.l)
            
            % mask for good enough spots
            mask_spots = spots(ii).fluo3.score > thresScore & ...
                spots(ii).fluo3.sigma > thresSigma(1) & ...
                spots(ii).fluo3.sigma < thresSigma(2);
            
            % only good spot long axis locations
            spots_l = spots(ii).fluo3.l(mask_spots);
            
            % choose cells with right number of MukBEF spots
            if ~isempty(spots_l) && sum(length(spots_l) == no_MukB) > 0
                % normalized spots location along long cell axis
                norm_dist = [norm_dist spots_l./spots(ii).length];
            end
        end
    end
    norm_dist_all = [norm_dist_all norm_dist];
    dist_all_means = [dist_all_means mean(abs(norm_dist))];
    
    % mean distances on negative and positive side
    fprintf('\n')
    disp(['# results ' num2str(kk) ' #'])
    disp(['Mean normalized distance from cell center - ' num2str((mean(abs(norm_dist)))) ...
        ', no. cells = ' num2str(length(norm_dist))])
    fprintf('\n')
end

fprintf('\n')
disp('# All results #')
disp(['Mean of repeats - ' num2str(mean(dist_all_means)) ...
    ', std ' num2str(std(dist_all_means)) ...
    ', no. cells = ' num2str(length(norm_dist_all))])
fprintf('\n')

% plot histogram of normalized ori1 localization (-0.5 to 0.5)
figure;
[N, X] = hist(norm_dist_all, -0.5:binning:0.5);
N = N./sum(N);
bar(X,N)
xlabel('Normalized long cell axis')
ylabel('Probability density')

figure; plot(X,N,'LineWidth',1)
xlabel('Normalized long cell axis')
ylabel('Probability density')


